/***************************************************************************
 *
 * Copyright 2012 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "ilm_client.h"
#include "ilm_control.h"
#include "LMControl.h"
#include "Expression.h"
#include "ExpressionInterpreter.h"
#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include <map>
#include <algorithm>
#include <iterator>
#include <cstring>
#include <signal.h> // signal
#include <unistd.h> // alarm

using namespace std;


#define COMMAND(text) COMMAND2(__COUNTER__,text)

#define COMMAND2(x,y) COMMAND3(x,y)

#define COMMAND3(funcNumber, text) \
    ilmErrorTypes func_ ## funcNumber(Expression* input); \
    static const bool reg_ ## funcNumber = \
        ExpressionInterpreter::addExpression(func_ ## funcNumber, text); \
    ilmErrorTypes func_ ## funcNumber(Expression* input)



//=============================================================================
COMMAND("help")
//=============================================================================
{
    (void)input;
    cout << "help: supported commands:\n\n";
    ExpressionInterpreter::printExpressionList();
    cout << "\n";

    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("tree")
//=============================================================================
{
    (void)input;
    cout << "help: supported commands:\n\n";
    ExpressionInterpreter::printExpressionTree();
    cout << "\n";

    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("get scene|screens|layers|surfaces")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("scene"))
    {
        callResult = printScene();
    }
    else if (input->contains("screens"))
    {
        (void)input;
        unsigned int count = 0;
        unsigned int* array = NULL;

        callResult = ilm_getScreenIDs(&count, &array);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get screen IDs\n";
            return callResult;
        }

        printArray("Screen", array, count);
    }
    else if (input->contains("layers"))
    {
        (void)input;
        int count = 0;
        unsigned int* array = NULL;

        callResult = ilm_getLayerIDs(&count, &array);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get layer IDs\n";
            return callResult;
        }

        printArray("Layer", array, count);
    }
    else if (input->contains("surfaces"))
    {
        (void)input;
        int count = 0;
        unsigned int* array = NULL;

        callResult = ilm_getSurfaceIDs(&count, &array);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get surface IDs\n";
            return callResult;
        }

        printArray("Surface", array, count);
    }

    return callResult;
}

//=============================================================================
COMMAND("get screen|layer|surface <id>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("screen"))
    {
        callResult = printScreenProperties(input->getUint("id"));
    }
    else if (input->contains("layer"))
    {
        callResult = printLayerProperties(input->getUint("id"));
    }
    else if (input->contains("surface"))
    {
        callResult = printSurfaceProperties(input->getUint("id"));
    }

    return callResult;
}

//=============================================================================
COMMAND("dump screen|layer|surface <id> to <file>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("screen"))
    {
        callResult = ilm_takeScreenshot(input->getUint("id"),
                                                        input->getString("file").c_str());
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to take screenshot of screen with ID " << input->getUint("id") << "\n";
            return callResult;
        }
    }
    else if (input->contains("layer"))
    {
        callResult = ilm_takeLayerScreenshot(input->getString("file").c_str(),
                                                            input->getUint("id"));
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to take screenshot of layer with ID " << input->getUint("id") << "\n";
            return callResult;
        }
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_takeSurfaceScreenshot(input->getString("file").c_str(),
                                                                input->getUint("id"));
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to take screenshot of surface with ID " << input->getUint("id") << "\n";
            return callResult;
        }
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> source region <x> <y> <w> <h>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    t_ilm_uint id = input->getUint("id");
    t_ilm_uint x = input->getUint("x");
    t_ilm_uint y = input->getUint("y");
    t_ilm_uint w = input->getUint("w");
    t_ilm_uint h = input->getUint("h");

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetSourceRectangle(id, x, y, w, h);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set source rectangle (" << x << "," << y << ", " << w << ", " << h << ") for layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetSourceRectangle(id, x, y, w, h);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set source rectangle (" << x << ", " << y << ", " << w << ", " << h << ") for surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> destination region <x> <y> <w> <h>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    t_ilm_uint id = input->getUint("id");
    t_ilm_uint x = input->getUint("x");
    t_ilm_uint y = input->getUint("y");
    t_ilm_uint w = input->getUint("w");
    t_ilm_uint h = input->getUint("h");

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetDestinationRectangle(id, x, y, w, h);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set destination rectangle (" << x << ", " << y << ", " << w << ", " << h << ") for layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetDestinationRectangle(id, x, y, w, h);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set destination rectangle (" << x << ", " << y << ", " << w << ", " << h << ") for surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> opacity <opacity>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    t_ilm_uint id = input->getUint("id");
    double opacity = input->getDouble("opacity");

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetOpacity(id, opacity);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set opacity " << opacity << " for layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetOpacity(id, opacity);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set opacity " << opacity << " for surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> visibility <visibility>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    t_ilm_uint id = input->getUint("id");
    t_ilm_bool visibility = input->getBool("visibility");

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetVisibility(id, visibility);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set visibility " << visibility << " for layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetVisibility(id, visibility);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set visibility " << visibility << " for surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> orientation <orientation>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    t_ilm_uint id = input->getUint("id");
    ilmOrientation orientation = (ilmOrientation)input->getInt("orientation");

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetOrientation(id, orientation);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set orientation " << orientation << " for layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetOrientation(id, orientation);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set orientation " << orientation << " for surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set screen|layer <id> render order [<idarray>]")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("screen"))
    {
        if (input->contains("idarray"))
        {
            unsigned int count = 0;
            unsigned int* array = NULL;
            unsigned int screenid = input->getUint("id");
            input->getUintArray("idarray", &array, &count);

            callResult = ilm_displaySetRenderOrder(screenid, array, count);
            if (ILM_SUCCESS != callResult)
            {
                cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
                cout << "Failed to set render order for screen with ID " << screenid << "\n";
                return callResult;
            }

            callResult = ilm_commitChanges();
        }
        else
        {
            unsigned int screenid = input->getUint("id");

            callResult = ilm_displaySetRenderOrder(screenid, NULL, 0);
            if (ILM_SUCCESS != callResult)
            {
                cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
                cout << "Failed to set render order for screen with ID " << screenid << "\n";
                return callResult;
            }

            callResult = ilm_commitChanges();
        }
    }
    else if (input->contains("layer"))
    {
        if (input->contains("idarray"))
        {
            unsigned int count = 0;
            unsigned int* array = NULL;
            unsigned int layerid = input->getUint("id");
            input->getUintArray("idarray", &array, &count);

            callResult = ilm_layerSetRenderOrder(layerid, array, count);
            if (ILM_SUCCESS != callResult)
            {
                cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
                cout << "Failed to set render order for layer with ID " << layerid << "\n";
                return callResult;
            }

            callResult = ilm_commitChanges();
        }
        else
        {
            unsigned int layerid = input->getUint("id");

            callResult = ilm_layerSetRenderOrder(layerid, NULL, 0);
            if (ILM_SUCCESS != callResult)
            {
                cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
                cout << "Failed to set render order for layer with ID " << layerid << "\n";
                return callResult;
            }

            callResult = ilm_commitChanges();
        }
    }

    return callResult;
}

//==============================================
COMMAND("set screen <id> gamma [<gamma_value>]")
//==============================================
{
	ilmErrorTypes callResult = ILM_FAILED;
	unsigned int screenid = input->getUint("id");
	double gamma = input->getDouble("gamma_value");

	callResult = ilm_displaySetGamma(screenid, gamma);
	if (ILM_SUCCESS != callResult)
	{
		cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
		cout << "Failed to set Gamma Value" << "\n";
		return callResult;
	}

	return callResult;
}


//==================================================================================================================================================
COMMAND("set screen <id> csc [<set_to_default>] [<hue>] [<saturation>] [<brightness>] [<contrast>] [<hue_off>] [<sat_off>] [<bright_off>] [<type>]")
//==================================================================================================================================================
{
	unsigned int screenid = input->getUint("id");
	ilmErrorTypes callResult = ILM_FAILED;
	ilmCSCProperties clrprop;

	clrprop.set_to_default = input->getBool("set_to_default");
	clrprop.hue = input->getInt("hue");
	clrprop.saturation = input->getInt("saturation");
	clrprop.brightness = input->getInt("brightness");
	clrprop.contrast = input->getInt("contrast");
	clrprop.hue_off = input->getInt("hue_off");
	clrprop.saturation_off = input->getInt("sat_off");
	clrprop.brightness_off = input->getInt("bright_off");
	clrprop.type = (ilmCSCTypes)input->getUint("type");

	callResult = ilm_displaySetCSC(screenid, &clrprop);
	if (ILM_SUCCESS != callResult)
	{
		cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
		cout << "Failed to set CSC matrix" << "\n";
		return callResult;
	}

	return callResult;
}

//=============================================================================
COMMAND("set synchronized surfaces [<idarray>]")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("idarray"))
    {
        unsigned int count = 0;
        unsigned int* array = NULL;

        input->getUintArray("idarray", &array, &count);

        callResult = ilm_setSynchronizedSurfaces(array, count);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set synchronized surface " << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("remove synchronized surfaces [<idarray>]")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("idarray"))
    {
        unsigned int count = 0;
        unsigned int* array = NULL;

        input->getUintArray("idarray", &array, &count);

        callResult = ilm_removeSynchronizedSurfaces(array, count);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set synchronized surface " << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> width <width>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("layer"))
    {
        unsigned int dimension[2];
        unsigned int layerid = input->getUint("id");

        callResult = ilm_layerGetDimension(layerid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get dimensions of layer with ID " << layerid << "\n";
            return callResult;
        }

        dimension[0] = input->getUint("width");

        callResult = ilm_layerSetDimension(layerid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set dimensions of layer with ID " << layerid << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        unsigned int dimension[2];
        unsigned int surfaceid = input->getUint("id");

        callResult = ilm_surfaceGetDimension(surfaceid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get dimensions of surface with ID " << surfaceid << "\n";
            return callResult;
        }

        dimension[0] = input->getUint("width");

        callResult = ilm_surfaceSetDimension(surfaceid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set dimensions of surface with ID " << surfaceid << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> height <height>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("layer"))
    {
        unsigned int dimension[2];
        unsigned int layerid = input->getUint("id");

        callResult = ilm_layerGetDimension(layerid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get dimensions of layer with ID " << layerid << "\n";
            return callResult;
        }

        dimension[1] = input->getUint("height");

        callResult = ilm_layerSetDimension(layerid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set dimensions of layer with ID " << layerid << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        unsigned int dimension[2];
        unsigned int surfaceid = input->getUint("id");

        callResult = ilm_surfaceGetDimension(surfaceid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to get dimensions of surface with ID " << surfaceid << "\n";
            return callResult;
        }

        dimension[1] = input->getUint("height");

        callResult = ilm_surfaceSetDimension(surfaceid, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set dimensions of surface with ID " << surfaceid << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("set layer|surface <id> position <x> <y>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    unsigned int id = input->getUint("id");
    unsigned int dimension[2];
    dimension[0] = input->getUint("x");
    dimension[1] = input->getUint("y");

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetPosition(id, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set position of layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetPosition(id, dimension);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set position of surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }

    return callResult;
}

//=============================================================================
COMMAND("create layer <layerid> <width> <height>")
//=============================================================================
{
    unsigned int layerid = input->getUint("layerid");
    unsigned int width = input->getUint("width");
    unsigned int height = input->getUint("height");

    ilmErrorTypes callResult = ilm_layerCreateWithDimension(&layerid, width, height);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to create layer with ID " << layerid << "\n";
    }

    return callResult;
}

//=============================================================================
COMMAND("create surface <surfaceid> <nativehandle> <width> <height> <pixelformat>")
//=============================================================================
{
    unsigned int surfaceid = input->getUint("surfaceid");
    unsigned int nativeHandle = input->getUint("nativehandle");
    unsigned int width = input->getUint("width");
    unsigned int height = input->getUint("height");
    e_ilmPixelFormat pixelformat = (e_ilmPixelFormat)input->getUint("pixelformat");

    ilmErrorTypes callResult = ilm_surfaceCreate(nativeHandle, width, height, pixelformat, &surfaceid);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to create surface with ID " << surfaceid << "\n";
    }

    return callResult;
}

//=============================================================================
COMMAND("destroy layer|surface <id>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("layer"))
    {
        unsigned int layerid = input->getUint("id");

        callResult = ilm_layerRemove(layerid);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to remove layer with ID " << layerid << "\n";
            return callResult;
        }
    }
    else if (input->contains("surface"))
    {
        unsigned int surfaceid = input->getUint("id");

        callResult = ilm_surfaceRemove(surfaceid);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to remove surface with ID " << surfaceid << "\n";
            return callResult;
        }
    }

    callResult = ilm_commitChanges();
    return callResult;
}

//=============================================================================
COMMAND("get communicator performance")
//=============================================================================
{
    (void) input; //suppress warning: unused parameter
    return getCommunicatorPerformance();
}

//=============================================================================
COMMAND("set surface <surfaceid> keyboard focus")
//=============================================================================
{
    t_ilm_surface surface = input->getUint("surfaceid");

    return setSurfaceKeyboardFocus(surface);
}

//=============================================================================
COMMAND("get keyboard focus")
//=============================================================================
{
    (void) input; //suppress warning: unused parameter
    return getKeyboardFocus();
}

//=============================================================================
COMMAND("set surface <surfaceid> pointer focus")
//=============================================================================
{
    t_ilm_surface surface = input->getUint("surfaceid");

    return setSurfacePointerFocus(surface);
}

//=============================================================================
COMMAND("get pointer focus")
//=============================================================================
{
    (void) input; //suppress warning: unused parameter
    return getPointerFocus();
}

//=============================================================================
COMMAND("set surface <surfaceid> accept <acceptance> input events from devices <kbd:pointer:touch>")
//=============================================================================
{
    t_ilm_surface surfaceId = input->getUint("surfaceid");
    t_ilm_bool acceptance = input->getBool("acceptance");
    string kbdPointerTouch = input->getString("kbd:pointer:touch");

    return setSurfaceAcceptsInput(surfaceId, kbdPointerTouch, acceptance);
}

//=============================================================================
COMMAND("set layer|surface <id> chromakey <red> <green> <blue>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;
    t_ilm_layer id = input->getUint("id");
    t_ilm_int color[3] =
    {
        input->getInt("red"),
        input->getInt("green"),
        input->getInt("blue")
    };

    if (input->contains("layer"))
    {
        callResult = ilm_layerSetChromaKey(id, color);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set chroma key (" << color[0] << ", " << color[1] << ", " << color[2] << ") for layer with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }    else if (input->contains("surface"))
    {
        callResult = ilm_surfaceSetChromaKey(id, color);
        if (ILM_SUCCESS != callResult)
        {
            cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
            cout << "Failed to set chroma key (" << color[0] << ", " << color[1] << ", " << color[2] << ") for surface with ID " << id << "\n";
            return callResult;
        }

        callResult = ilm_commitChanges();
    }
    return callResult;
}

//=============================================================================
COMMAND("set surface <surfaceid> chromakey disabled")
//=============================================================================
{
    t_ilm_surface surface = input->getUint("surfaceid");

    ilmErrorTypes callResult = ilm_surfaceSetChromaKey(surface, NULL);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to disable chroma key for surface with ID " << surface << "\n";
    }
    else
    {
        callResult = ilm_commitChanges();
    }
    return callResult;
}

//=============================================================================
COMMAND("test notification layer <layerid>")
//=============================================================================
{
    unsigned int layerid = input->getUint("layerid");

    return testNotificationLayer(layerid);
}

//=============================================================================
COMMAND("watch layer|surface <idarray>")
//=============================================================================
{
    ilmErrorTypes callResult = ILM_FAILED;

    if (input->contains("layer"))
    {
        unsigned int* layerids = NULL;
        unsigned int layeridCount;
        input->getUintArray("idarray", &layerids, &layeridCount);

        callResult = watchLayer(layerids, layeridCount);
    }
    else if (input->contains("surface"))
    {
        unsigned int* surfaceids = NULL;
        unsigned int surfaceidCount;
        input->getUintArray("idarray", &surfaceids, &surfaceidCount);

        callResult = watchSurface(surfaceids, surfaceidCount);
    }
    return callResult;
}

//=============================================================================
COMMAND("set optimization <id> mode <mode>")
//=============================================================================
{
    t_ilm_uint id = input->getUint("id");
    t_ilm_uint mode = input->getUint("mode");
    return setOptimization(id, mode);
}

//=============================================================================
COMMAND("get optimization <id>")
//=============================================================================
{
    t_ilm_uint id = input->getUint("id");
    return getOptimization(id);
}

//=============================================================================
COMMAND("analyze surface <surfaceid>")
//=============================================================================
{
    t_ilm_surface targetSurfaceId = (t_ilm_uint) input->getUint("surfaceid");
    analyzeSurface(targetSurfaceId);
    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("scatter [all]")
//=============================================================================
{
    if (input->contains("all"))
    {
        scatterAll();
    }
    else
    {
        scatter();
    }
    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("demo [<animation_mode=2>]")
//=============================================================================
{
    t_ilm_uint mode = (t_ilm_uint) input->getUint("animation_mode");
    demo(mode);
    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("export scene to <filename>")
//=============================================================================
{
    string filename = (string) input->getString("filename");
    exportSceneToFile(filename);
    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("export xtext to <filename> <grammar> <url>")
//=============================================================================
{
    string filename = (string) input->getString("filename");
    string grammar = (string) input->getString("grammar");
    string url = (string) input->getString("url");
    exportXtext(filename, grammar, url);
    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("import scene from <filename>")
//=============================================================================
{
    string filename = (string) input->getString("filename");
    importSceneFromFile(filename);
    return ILM_SUCCESS;
}

//=============================================================================
COMMAND("surface <surfaceid> set seat acceptance [<strarray>]")
//=============================================================================
{
    unsigned int count = 0;
    char** strArray = NULL;
    ilmErrorTypes callResult;
    t_ilm_surface SurfaceId = input->getUint("surfaceid");
    input->getStringArray("strarray", &strArray, &count);
    callResult = ilm_setInputAcceptanceOn(SurfaceId, count, strArray);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to set accepted seats: ";
        for (unsigned int strCnt = 0; strCnt < count; strCnt++)
        {
            std::string seatName(strArray[strCnt]);
            cout << seatName << ", ";
        }
        cout << ":for surface with ID " << SurfaceId << "\n";
    }
    if ((NULL != strArray) && (count > 0))
    {
        for (unsigned int strCnt = 0; strCnt < count; strCnt++)
        {
            if(NULL != strArray[strCnt])
            {
                free(strArray[strCnt]);
            }
        }
        delete strArray;
    }
    return callResult;
}

//=============================================================================
COMMAND("surface <surfaceid> get seat acceptance")
//=============================================================================
{
    unsigned int count = 0;
    char** strArray = NULL;
    ilmErrorTypes callResult;
    t_ilm_surface SurfaceId = input->getUint("surfaceid");
    callResult = ilm_getInputAcceptanceOn(SurfaceId, &count, &strArray);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to get accepted seats for surface: " << SurfaceId;
    }
    else
    {
        printStrArray("Accepts Inputs from Seats ", strArray, count);
    }
    if ((NULL != strArray) && (count > 0))
    {
        for (unsigned int strCnt = 0; strCnt < count; strCnt++)
        {
            if(NULL != strArray[strCnt])
            {
                free(strArray[strCnt]);
            }
        }
        free(strArray);
    }
    return callResult;
}

//=============================================================================
COMMAND("get input devices <kbd:pointer:touch>")
//=============================================================================
{
    string kbdPointerTouch = input->getString("kbd:pointer:touch");
    char* str;
    char* tok;
    unsigned int count = 0;
    char** strArray = NULL;
    ilmErrorTypes callResult;
    ilmInputDevice devices = (ilmInputDevice)0;

    str = new char [kbdPointerTouch.size()+1];
    strcpy(str, kbdPointerTouch.c_str());
    tok = strtok(str, ":");
    while (tok != NULL)
    {
        if (!strcmp(tok, "kbd"))
        {
            devices |= ILM_INPUT_DEVICE_KEYBOARD;
        }
        else if (!strcmp(tok, "pointer"))
        {
            devices |= ILM_INPUT_DEVICE_POINTER;
        }
        else if (!strcmp(tok, "touch"))
        {
            devices |= ILM_INPUT_DEVICE_TOUCH;
        }
        else
        {
            cerr << "Unknown devices specified." << endl;
        }

        tok = strtok(NULL, ":");
    }
    callResult = ilm_getInputDevices(devices, &count, &strArray);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to get seats supporting " << kbdPointerTouch;
    }
    else
    {
        cout << "Seats having " << kbdPointerTouch << " device: \n";
        printStrArray("", strArray, count);
    }
    if ((NULL != strArray) && (count > 0))
    {
        for (unsigned int strCnt = 0; strCnt < count; strCnt++)
        {
            if(NULL != strArray[strCnt])
            {
                free(strArray[strCnt]);
            }
        }
        free(strArray);
    }
    return callResult;
}


//=============================================================================
COMMAND("get device capabilities <seatname>")
//=============================================================================
{
    string seatName = input->getString("seatname");
    ilmErrorTypes callResult;
    ilmInputDevice devices = (ilmInputDevice)0;

    callResult = ilm_getInputDeviceCapabilities((t_ilm_string)seatName.c_str(), &devices);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to get devices supported by " << seatName;
    }
    else
    {
        cout << "Devices supported by " << seatName << " are: "
                << (devices & ILM_INPUT_DEVICE_KEYBOARD ? "keyboard " : "")
                << (devices & ILM_INPUT_DEVICE_POINTER ? "mouse " : "")
                << (devices & ILM_INPUT_DEVICE_TOUCH ? "touch " : "")
                << "\n";
    }
    return callResult;
}

//=============================================================================
COMMAND("set input focus <surfaceid> <kbd:pointer:touch> <focus>")
//=============================================================================
{
    t_ilm_surface SurfaceId = input->getUint("surfaceid");
    string kbdPointerTouch = input->getString("kbd:pointer:touch");
    t_ilm_bool focus = input->getBool("focus");
    ilmErrorTypes callResult;
    char* str;
    char* tok;
    ilmInputDevice devices = (ilmInputDevice)0;

    str = new char [kbdPointerTouch.size()+1];
    strcpy(str, kbdPointerTouch.c_str());
    tok = strtok(str, ":");
    while (tok != NULL)
    {
        if (!strcmp(tok, "kbd"))
        {
            devices |= ILM_INPUT_DEVICE_KEYBOARD;
        }
        else if (!strcmp(tok, "pointer"))
        {
            devices |= ILM_INPUT_DEVICE_POINTER;
        }
        else if (!strcmp(tok, "touch"))
        {
            devices |= ILM_INPUT_DEVICE_TOUCH;
        }
        else
        {
            cerr << "Unknown devices specified." << endl;
        }

        tok = strtok(NULL, ":");
    }

    callResult = ilm_setInputFocus(&SurfaceId, 1, devices, focus);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to set " << kbdPointerTouch << " focus to " << focus << " on surface " << SurfaceId;
    }
    return callResult;
}

//=============================================================================
COMMAND("get all input focuses")
//=============================================================================
{
    t_ilm_surface *pSurfaceId = NULL;
    ilmInputDevice *pDevice = NULL;
    t_ilm_uint numSurfaces = 0;
    ilmErrorTypes callResult;
    callResult = ilm_getInputFocus(&pSurfaceId, &pDevice, &numSurfaces);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to get input focus of surfaces\n";
    }
    else
    {
        for (unsigned int surf = 0; surf < numSurfaces; surf++)
        {
            cout << "Surface " << pSurfaceId[surf] << " has :"
                    << (pDevice[surf] & ILM_INPUT_DEVICE_KEYBOARD ? "keyboard " : "")
                    << (pDevice[surf] & ILM_INPUT_DEVICE_POINTER ? "mouse " : "")
                    << (pDevice[surf] & ILM_INPUT_DEVICE_TOUCH ? "touch " : "")
                    << "focus\n";

        }
        if (NULL != pSurfaceId)
        {
            free (pSurfaceId);
        }
        if (NULL != pDevice)
        {
            free (pDevice);
        }
    }
    return callResult;
}

//=============================================================================
COMMAND("set screen <id> state <on:standby:suspend:off>")
//=============================================================================
{
    unsigned int screenID = input->getUint("id");
    string screenState = input->getString("on:standby:suspend:off");
    ilmErrorTypes callResult = ILM_SUCCESS;
    char* str;
    char* tok;
    ilmScreenState lmScreenState = ILM_SCREEN_UNKNOWN;
    str = new char [screenState.size()+1];
    strcpy(str, screenState.c_str());
    tok = strtok(str, ":");
    while (tok != NULL)
    {
        if (!strcmp(tok, "on"))
        {
        	lmScreenState = ILM_SCREEN_ON;
        }
        else if (!strcmp(tok, "standby"))
        {
        	lmScreenState = ILM_SCREEN_STANDBY;
        }
        else if (!strcmp(tok, "suspend"))
        {
        	lmScreenState = ILM_SCREEN_SUSPEND;
        }
        else if (!strcmp(tok, "off"))
        {
        	lmScreenState = ILM_SCREEN_OFF;
        }
        else
        {
        	cout << "Unknown screen state passed\n";
        	callResult = ILM_ERROR_INVALID_ARGUMENTS;
        }
        tok = strtok(NULL, ":");
    }
    if (ILM_SUCCESS == callResult)
    {
    	cout << "ilm_screenSetState(screenID=" << screenID;
    	callResult = ilm_screenSetState(screenID, lmScreenState);
    }
    return callResult;
}


//=============================================================================
COMMAND("get screen <id> state")
//=============================================================================
{
    unsigned int screenID = input->getUint("id");
    ilmErrorTypes callResult = ILM_SUCCESS;
    unsigned int lmScreenState = ILM_SCREEN_UNKNOWN;
    callResult = ilm_screenGetState(screenID, &lmScreenState);
    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to get state of screen id: " << screenID << "\n";
    }
    else
    {
       if (lmScreenState == ILM_SCREEN_ON)
       {
            cout << "Screen id:" << screenID <<" state is ON \n";
       }
       else if (lmScreenState == ILM_SCREEN_STANDBY)
       {
            cout << "Screen id:" << screenID <<" state is STANDBY \n";
       }
       else if (lmScreenState == ILM_SCREEN_SUSPEND)
       {
            cout << "Screen id:" << screenID <<" state is SUSPEND \n";
       }
       else if (lmScreenState == ILM_SCREEN_OFF)
       {
            cout << "Screen id:" << screenID <<" state is OFF \n";
       }
    }
    return callResult;
}

//=============================================================================
COMMAND("set screen <id> alpha <alphavalue>")
//=============================================================================
{
    unsigned int screenID = input->getUint("id");
    unsigned int alphaValue = input->getUint("alphavalue");;
    ilmErrorTypes callResult = ILM_SUCCESS;

    if ((0 <= alphaValue) && (alphaValue < 256))
    {
        callResult = ilm_screenSetAlpha(screenID, alphaValue);
    }
    else
    {
        cout << "setting alpha value for screenID : " << screenID << " FAILED \n";
        cout << "set alpha value between 0 to 255 ";
        callResult = ILM_ERROR_INVALID_ARGUMENTS;
    }

    return callResult;
}

//=============================================================================
COMMAND("get screen <id> alpha")
//=============================================================================
{
    unsigned int screenID = input->getUint("id");
    unsigned int alphaValue = 0;
    ilmErrorTypes callResult = ILM_SUCCESS;
    callResult = ilm_screenGetAlpha(screenID, &alphaValue);

    if (ILM_SUCCESS != callResult)
    {
        cout << "LayerManagerService returned: " << ILM_ERROR_STRING(callResult) << "\n";
        cout << "Failed to get alpha of screen id: " << screenID << "\n";
    }
    else
    {
        cout << "Screen id:" << screenID <<" alpha value is " << alphaValue << "\n";
    }

    return callResult;
}

